// PDP-8 Emulator for SymbOS, version 1.0
// Copyright 2025 Daniel E. Gaskell
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the Software), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED AS IS, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include <symbos.h>
#include <graphics.h>
#include <string.h>

#include "cpu.h"
#include "assets.c"
#define LINE_LEN 60
#define LINES 30
#define CURSOR_ID 77
#define BUF_SIZE 128
#define TABWIDTH 8

// To fit everything into memory without allowing any canvases to cross page
// boundaries, the memory map is a bit weird and splits the main canvas into 3.
_data char canvas1[232*89/2 + 24];
_transfer char canvas2[232*43/2 + 24];
_transfer char canvas3[28*132/2 + 24];
_transfer char holecanvas[(8*14/2) + 24];
_transfer char text[LINES][LINE_LEN+1];
unsigned char tape_in = 8;
unsigned char tape_out = 8;
char countbyte = 0;
char active = 0;
char toggle_back = 0;

// buffers that overlay feedsimg[] in assets.c
char* tape_in_buf = feedsimg + sizeof(feedsimg) - BUF_SIZE;
char* tape_out_buf = feedsimg + sizeof(feedsimg) - BUF_SIZE*2;
char* keybuf = feedsimg + sizeof(feedsimg) - BUF_SIZE*3;
char* keybufptr = feedsimg + sizeof(feedsimg) - BUF_SIZE*3;
char* tape_in_end = feedsimg + sizeof(feedsimg);
char* tape_in_ptr;
char* tape_out_end = feedsimg + sizeof(feedsimg) - BUF_SIZE;
char* tape_out_ptr;

// main form data
_transfer Ctrl c_canvas1 = {0, C_IMAGE_EXT, -1, (unsigned short)canvas1,   0,  0, 232,  89};
_transfer Ctrl c_canvas2 = {0, C_IMAGE_EXT, -1, (unsigned short)canvas2,   0, 89, 232,  43};
_transfer Ctrl c_canvas3 = {1, C_IMAGE_EXT, -1, (unsigned short)canvas3, 232,  0,  28, 132};
_transfer Ctrl_Group ctrls = {3, 0, &c_canvas1};

_transfer Window form1 = {
    WIN_NORMAL,
    WIN_CLOSE | WIN_TITLE | WIN_ICON,
    0,          // PID
    10, 10,     // x, y
    260, 132,   // w, h
    0, 0,       // xscroll, yscroll
    260, 132,   // wfull, hfull
    260, 132,   // wmin, hmin
    260, 132,   // wmax, hmax
    icon1,      // icon
    "PDP-8 Emulator", // title text
    0,          // no status text
    0,          // no menu
    &ctrls};    // controls

// control extended data
_transfer Ctrl_Text_Font cd_line1 = {&text[0][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line2 = {&text[1][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line3 = {&text[2][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line4 = {&text[3][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line5 = {&text[4][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line6 = {&text[5][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line7 = {&text[6][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line8 = {&text[7][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line9 = {&text[8][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line10 = {&text[9][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line11 = {&text[10][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line12 = {&text[11][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line13 = {&text[12][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line14 = {&text[13][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line15 = {&text[14][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line16 = {&text[15][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line17 = {&text[16][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line18 = {&text[17][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line19 = {&text[18][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line20 = {&text[19][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line21 = {&text[20][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line22 = {&text[21][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line23 = {&text[22][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line24 = {&text[23][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line25 = {&text[24][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line26 = {&text[25][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line27 = {&text[26][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line28 = {&text[27][0], (COLOR_BLACK << 4) | COLOR_WHITE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line29 = {&text[28][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};
_transfer Ctrl_Text_Font cd_line30 = {&text[29][0], (COLOR_BLACK << 4) | COLOR_LBLUE, ALIGN_LEFT | TEXT_16COLOR, font};

// control data
_transfer Ctrl c_hole1    = {1,  C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, -98, 8, 14};
_transfer Ctrl c_stripe1  = {2,  C_AREA,      -1, COLOR_LBLUE | AREA_16COLOR, 8, -98, 208, 14};
_transfer Ctrl c_hole2    = {3,  C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, -98, 8, 14};
_transfer Ctrl c_line1    = {4,  C_TEXT_FONT, -1, (unsigned short)&cd_line1,  10, -98, 208, 7};
_transfer Ctrl c_line2    = {5,  C_TEXT_CTRL, -1, (unsigned short)&cd_line2,  10, -91, 208, 7};
_transfer Ctrl c_hole3    = {6,  C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, -84, 8, 14};
_transfer Ctrl c_stripe3  = {7,  C_AREA,      -1, COLOR_WHITE | AREA_16COLOR, 8, -84, 208, 14};
_transfer Ctrl c_hole4    = {8,  C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, -84, 8, 14};
_transfer Ctrl c_line3    = {9,  C_TEXT_FONT, -1, (unsigned short)&cd_line3,  10, -84, 208, 7};
_transfer Ctrl c_line4    = {10, C_TEXT_FONT, -1, (unsigned short)&cd_line4,  10, -77, 208, 7};
_transfer Ctrl c_hole5    = {11, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, -70, 8, 14};
_transfer Ctrl c_stripe5  = {12, C_AREA,      -1, COLOR_LBLUE | AREA_16COLOR, 8, -70, 208, 14};
_transfer Ctrl c_hole6    = {13, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, -70, 8, 14};
_transfer Ctrl c_line5    = {14, C_TEXT_FONT, -1, (unsigned short)&cd_line5,  10, -70, 208, 7};
_transfer Ctrl c_line6    = {15, C_TEXT_FONT, -1, (unsigned short)&cd_line6,  10, -63, 208, 7};
_transfer Ctrl c_hole7    = {16, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, -56, 8, 14};
_transfer Ctrl c_stripe7  = {17, C_AREA,      -1, COLOR_WHITE | AREA_16COLOR, 8, -56, 208, 14};
_transfer Ctrl c_hole8    = {18, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, -56, 8, 14};
_transfer Ctrl c_line7    = {19, C_TEXT_FONT, -1, (unsigned short)&cd_line7,  10, -56, 208, 7};
_transfer Ctrl c_line8    = {20, C_TEXT_FONT, -1, (unsigned short)&cd_line8,  10, -49, 208, 7};
_transfer Ctrl c_hole9    = {21, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, -42, 8, 14};
_transfer Ctrl c_stripe9  = {22, C_AREA,      -1, COLOR_LBLUE | AREA_16COLOR, 8, -42, 208, 14};
_transfer Ctrl c_hole10   = {23, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, -42, 8, 14};
_transfer Ctrl c_line9    = {24, C_TEXT_FONT, -1, (unsigned short)&cd_line9,  10, -42, 208, 7};
_transfer Ctrl c_line10   = {25, C_TEXT_FONT, -1, (unsigned short)&cd_line10, 10, -35, 208, 7};
_transfer Ctrl c_hole11   = {26, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, -28, 8, 14};
_transfer Ctrl c_stripe11 = {27, C_AREA,      -1, COLOR_WHITE | AREA_16COLOR, 8, -28, 208, 14};
_transfer Ctrl c_hole12   = {28, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, -28, 8, 14};
_transfer Ctrl c_line11   = {29, C_TEXT_FONT, -1, (unsigned short)&cd_line11, 10, -28, 208, 7};
_transfer Ctrl c_line12   = {30, C_TEXT_FONT, -1, (unsigned short)&cd_line12, 10, -21, 208, 7};
_transfer Ctrl c_hole13   = {31, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, -14, 8, 14};
_transfer Ctrl c_stripe13 = {32, C_AREA,      -1, COLOR_LBLUE | AREA_16COLOR, 8, -14, 208, 14};
_transfer Ctrl c_hole14   = {33, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, -14, 8, 14};
_transfer Ctrl c_line13   = {34, C_TEXT_FONT, -1, (unsigned short)&cd_line13, 10, -14, 208, 7};
_transfer Ctrl c_line14   = {35, C_TEXT_FONT, -1, (unsigned short)&cd_line14, 10, -7, 208, 7};
_transfer Ctrl c_hole15   = {36, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, 0, 8, 14};
_transfer Ctrl c_stripe15 = {37, C_AREA,      -1, COLOR_WHITE | AREA_16COLOR, 8, 0, 208, 14};
_transfer Ctrl c_hole16   = {38, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, 0, 8, 14};
_transfer Ctrl c_line15   = {39, C_TEXT_FONT, -1, (unsigned short)&cd_line15, 10, 0, 208, 7};
_transfer Ctrl c_line16   = {40, C_TEXT_FONT, -1, (unsigned short)&cd_line16, 10, 7, 208, 7};
_transfer Ctrl c_hole17   = {41, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, 14, 8, 14};
_transfer Ctrl c_stripe17 = {42, C_AREA,      -1, COLOR_LBLUE | AREA_16COLOR, 8, 14, 208, 14};
_transfer Ctrl c_hole18   = {43, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, 14, 8, 14};
_transfer Ctrl c_line17   = {44, C_TEXT_FONT, -1, (unsigned short)&cd_line17, 10, 14, 208, 7};
_transfer Ctrl c_line18   = {45, C_TEXT_FONT, -1, (unsigned short)&cd_line18, 10, 21, 208, 7};
_transfer Ctrl c_hole19   = {46, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, 28, 8, 14};
_transfer Ctrl c_stripe19 = {47, C_AREA,      -1, COLOR_WHITE | AREA_16COLOR, 8, 28, 208, 14};
_transfer Ctrl c_hole20   = {48, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, 28, 8, 14};
_transfer Ctrl c_line19   = {49, C_TEXT_FONT, -1, (unsigned short)&cd_line19, 10, 28, 208, 7};
_transfer Ctrl c_line20   = {50, C_TEXT_FONT, -1, (unsigned short)&cd_line20, 10, 35, 208, 7};
_transfer Ctrl c_hole21   = {51, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, 42, 8, 14};
_transfer Ctrl c_stripe21 = {52, C_AREA,      -1, COLOR_LBLUE | AREA_16COLOR, 8, 42, 208, 14};
_transfer Ctrl c_hole22   = {53, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, 42, 8, 14};
_transfer Ctrl c_line21   = {54, C_TEXT_FONT, -1, (unsigned short)&cd_line21, 10, 42, 208, 7};
_transfer Ctrl c_line22   = {55, C_TEXT_FONT, -1, (unsigned short)&cd_line22, 10, 49, 208, 7};
_transfer Ctrl c_hole23   = {56, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, 56, 8, 14};
_transfer Ctrl c_stripe23 = {57, C_AREA,      -1, COLOR_WHITE | AREA_16COLOR, 8, 56, 208, 14};
_transfer Ctrl c_hole24   = {58, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, 56, 8, 14};
_transfer Ctrl c_line23   = {59, C_TEXT_FONT, -1, (unsigned short)&cd_line23, 10, 56, 208, 7};
_transfer Ctrl c_line24   = {60, C_TEXT_FONT, -1, (unsigned short)&cd_line24, 10, 63, 208, 7};
_transfer Ctrl c_hole25   = {61, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, 70, 8, 14};
_transfer Ctrl c_stripe25 = {62, C_AREA,      -1, COLOR_LBLUE | AREA_16COLOR, 8, 70, 208, 14};
_transfer Ctrl c_hole26   = {63, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, 70, 8, 14};
_transfer Ctrl c_line25   = {64, C_TEXT_FONT, -1, (unsigned short)&cd_line25, 10, 70, 208, 7};
_transfer Ctrl c_line26   = {65, C_TEXT_FONT, -1, (unsigned short)&cd_line26, 10, 77, 208, 7};
_transfer Ctrl c_hole27   = {66, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, 84, 8, 14};
_transfer Ctrl c_stripe27 = {67, C_AREA,      -1, COLOR_WHITE | AREA_16COLOR, 8, 84, 208, 14};
_transfer Ctrl c_hole28   = {68, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, 84, 8, 14};
_transfer Ctrl c_line27   = {69, C_TEXT_FONT, -1, (unsigned short)&cd_line27, 10, 84, 208, 7};
_transfer Ctrl c_line28   = {70, C_TEXT_FONT, -1, (unsigned short)&cd_line28, 10, 91, 208, 7};
_transfer Ctrl c_hole29   = {71, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 0, 98, 8, 14};
_transfer Ctrl c_stripe29 = {72, C_AREA,      -1, COLOR_LBLUE | AREA_16COLOR, 8, 98, 208, 14};
_transfer Ctrl c_hole30   = {73, C_IMAGE_EXT, -1, (unsigned short)holecanvas, 216, 98, 8, 14};
_transfer Ctrl c_line29   = {74, C_TEXT_FONT, -1, (unsigned short)&cd_line29, 10, 98, 208, 7};
_transfer Ctrl c_line30   = {75, C_TEXT_FONT, -1, (unsigned short)&cd_line30, 10, 105, 208, 7};
_transfer Ctrl c_cursor1   = {76, C_AREA, -1, COLOR_BLACK, 7, 109, 3, 3};
_transfer Ctrl c_cursor2   = {CURSOR_ID, C_AREA, -1, COLOR_BLACK, 14, 109, 3, 3};

// control calculation group
_transfer Calc_Rule cc_hole1    = { 0, 0, 1, -210, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe1  = { 8, 0, 1, -210, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole2    = {-8, 1, 1, -210, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line1    = {10, 0, 1, -210, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line2    = {10, 0, 1, -203, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole3    = { 0, 0, 1, -196, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe3  = { 8, 0, 1, -196, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole4    = {-8, 1, 1, -196, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line3    = {10, 0, 1, -196, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line4    = {10, 0, 1, -189, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole5    = { 0, 0, 1, -182, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe5  = { 8, 0, 1, -182, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole6    = {-8, 1, 1, -182, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line5    = {10, 0, 1, -182, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line6    = {10, 0, 1, -175, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole7    = { 0, 0, 1, -168, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe7  = { 8, 0, 1, -168, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole8    = {-8, 1, 1, -168, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line7    = {10, 0, 1, -168, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line8    = {10, 0, 1, -161, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole9    = { 0, 0, 1, -154, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe9  = { 8, 0, 1, -154, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole10   = {-8, 1, 1, -154, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line9    = {10, 0, 1, -154, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line10   = {10, 0, 1, -147, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole11   = { 0, 0, 1, -140, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe11 = { 8, 0, 1, -140, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole12   = {-8, 1, 1, -140, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line11   = {10, 0, 1, -140, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line12   = {10, 0, 1, -133, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole13   = { 0, 0, 1, -126, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe13 = { 8, 0, 1, -126, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole14   = {-8, 1, 1, -126, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line13   = {10, 0, 1, -126, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line14   = {10, 0, 1, -119, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole15   = { 0, 0, 1, -112, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe15 = { 8, 0, 1, -112, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole16   = {-8, 1, 1, -112, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line15   = {10, 0, 1, -112, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line16   = {10, 0, 1, -105, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole17   = { 0, 0, 1,  -98, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe17 = { 8, 0, 1,  -98, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole18   = {-8, 1, 1,  -98, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line17   = {10, 0, 1,  -98, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line18   = {10, 0, 1,  -91, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole19   = { 0, 0, 1,  -84, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe19 = { 8, 0, 1,  -84, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole20   = {-8, 1, 1,  -84, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line19   = {10, 0, 1,  -84, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line20   = {10, 0, 1,  -77, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole21   = { 0, 0, 1,  -70, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe21 = { 8, 0, 1,  -70, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole22   = {-8, 1, 1,  -70, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line21   = {10, 0, 1,  -70, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line22   = {10, 0, 1,  -63, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole23   = { 0, 0, 1,  -56, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe23 = { 8, 0, 1,  -56, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole24   = {-8, 1, 1,  -56, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line23   = {10, 0, 1,  -56, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line24   = {10, 0, 1,  -49, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole25   = { 0, 0, 1,  -42, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe25 = { 8, 0, 1,  -42, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole26   = {-8, 1, 1,  -42, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line25   = {10, 0, 1,  -42, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line26   = {10, 0, 1,  -35, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole27   = { 0, 0, 1,  -28, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe27 = { 8, 0, 1,  -28, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole28   = {-8, 1, 1,  -28, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line27   = {10, 0, 1,  -28, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line28   = {10, 0, 1,  -21, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_hole29   = { 0, 0, 1,  -14, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_stripe29 = { 8, 0, 1,  -14, 1, 1, -16, 1, 1, 14, 0, 1};
_transfer Calc_Rule cc_hole30   = {-8, 1, 1,  -14, 1, 1,   8, 0, 1, 14, 0, 1};
_transfer Calc_Rule cc_line29   = {10, 0, 1,  -14, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_line30   = {10, 0, 1,   -7, 1, 1, -19, 1, 1,  7, 0, 1};
_transfer Calc_Rule cc_cursor1  = { 7, 0, 1,   -3, 1, 1,   3, 0, 1,  3, 0, 1};
_transfer Calc_Rule cc_cursor2  = {14, 0, 1,   -3, 1, 1,   3, 0, 1,  3, 0, 1};

_transfer Ctrl_Group ctrls2 = {CURSOR_ID, 0, &c_hole1, &cc_hole1};

_transfer Window form2 = {
    WIN_NORMAL,
    WIN_CLOSE | WIN_TITLE | WIN_ICON | WIN_RESIZABLE | WIN_ADJUSTX | WIN_ADJUSTY,
    0,          // PID
    60, 50,    // x, y
    224, 112,   // w, h
    0, 0,       // xscroll, yscroll
    224, 112,   // wfull, hfull
    119, 56,    // wmin, hmin
    320, 210,   // wmax, hmax
    icon2,      // icon
    "Teletype", // title text
    0,          // no status text
    0,          // no menu
    &ctrls2};   // controls

// splash form data
_transfer Ctrl_Text cd_loading = {"Loading...", (COLOR_BLACK << 2) | COLOR_ORANGE, ALIGN_CENTER};
_transfer Ctrl c_area = {0, C_AREA, -1, COLOR_ORANGE, 0, 0, 64, 24};
_transfer Ctrl c_loading = {1, C_TEXT, -1, (unsigned short)&cd_loading, 2, 8, 62, 8};
_transfer Ctrl_Group ctrls3 = {2, 0, &c_area};

_transfer Window form3 = {
    WIN_NORMAL,
    WIN_NOTTASKBAR | WIN_NOTMOVEABLE,
    0,          // PID
    0, 0,       // x, y
    64, 24,     // w, h
    0, 0,       // xscroll, yscroll
    64, 24,     // wfull, hfull
    64, 24,     // wmin, hmin
    64, 24,     // wmax, hmax
    0,          // no icon
    0,          // no title text
    0,          // no status text
    0,          // no menu
    &ctrls3};   // controls

// interface
char winID = 0;
char teleID = 0;
char splashID = 0;
char power = 0;
char lock = 0;
char run = 0;
char buttons[26];
char button_colors[26] = {0, 0, 0, 2, 2, 2, 0, 0, 0, 2, 2, 2, 0, 0, 0, 2, 2, 2, 0, 2, 2, 2, 2, 2, 0};
char inst_light = -1;
char state_lights = 0;
char tty_x = 0;
char tty_w = 41;
char tape_in_next = 0;  // tape reader: 0 = no change, 1 = insert tape, 2 = advance tape, 3 = unloading, 4 = fully unload
char tape_out_next = 0; // tape punch:  0 = no change, 1 = insert tape, 2 = advance tape,                4 = fully unload
char tape_in_loaded = 0;
unsigned short rndseed = 0xACE1;
unsigned char tty_scroll = 0;
unsigned char debug = 0;
unsigned short breakpoint = 0xFFFF;
unsigned short old_breakpoint, skip_breakpoint;

// rim loader
unsigned short rimldr[18] = {06014, 06011, 05357, 06016, 07106, 07006, 07510, 05357,
                             07006, 06011, 05367, 06016, 07420, 03776, 03376, 05357, 0, 0};

unsigned short binldr[] =   {00000, 03212, 04260, 01300, 07750, 05237, 02212, 07040,
                             05227, 01212, 07640, 05230, 01214, 00274, 01341, 07510,
                             02226, 07750, 05626, 01214, 00256, 01257, 03213, 05230,
                             00070, 06201, 00000, 00000, 06031, 05262, 06036, 03214,
                             01214, 05660, 06011, 05270, 06016, 05265, 00300, 04343,
                             07041, 01215, 07402, 06032, 06014, 06214, 01257, 03213,
                             07604, 07700, 01353, 01352, 03261, 04226, 05313, 03215,
                             01213, 03336, 01214, 03376, 04260, 03355, 04226, 05275,
                             04343, 07420, 05336, 03216, 01376, 01355, 01215, 05315,
                             00000, 03616, 02216, 07600, 05332, 00000, 01376, 07106,
                             07006, 07006, 01355, 05743, 05262, 00006, 00000, 00000};

// emulation
unsigned short acc, link, ir, ma, mb, mq, ion, ion_mmu;
unsigned char dfield, ifield, ibuf, intbuf;
unsigned short halt;
unsigned short pc = 1; // internal PC*2 (relative to zero; note that this points to high byte of opcode)
unsigned char pt_rbuf = 0;
unsigned char pt_pbuf = 0;
unsigned char pt_rflag = 0;
unsigned char pt_pflag = 0;
unsigned char pt_pgo = 0; // separate from pt_pflag because just resetting the punch flag with PCF does *not* initiate a print, PPC does!
unsigned char pt_rgo = 0; // separate from pt_rflag because we shouldn't set the flag until actually requested by RFC.
unsigned char pt_bin = 0;
unsigned char pt_rstage = 0;
unsigned char tty_kbuf = 0;
unsigned char tty_tbuf = 0;
unsigned char tty_kflag = 0;
unsigned char tty_tflag = 0;
unsigned char tty_tgo = 0; // separate from tty_tflag because just resetting the printer flag with TCF does *not* initiate a print, TPC does!
unsigned short old_acc, old_link, old_ma, old_mb, old_mq, old_ion, old_run;
unsigned short old_dfield, old_ifield, old_pc, old_inst_light, old_state_lights;

#define IRQ_PTR 1
#define IRQ_PTP 2
#define IRQ_TTYK 4
#define IRQ_TTYP 8
unsigned char irq = 0;

void clearflags(void) {
    acc = 0;
    link = 0;
    ion = 0;
    ion_mmu = 0;
    irq = 0;
    pt_rflag = 0;
    pt_pflag = 0;
    pt_pgo = 0;
    pt_rgo = 0;
    tty_kflag = 0;
    tty_tflag = 0;
    tty_tgo = 0;
}

// Note: Model 33 teletype codes are basically ASCII where it matters (CR, LF, TAB, BS, DEL, ESC)
void tty_out(unsigned char ch) {
    unsigned char i, color;
    unsigned char scroll = 0;
    signed short refresh_y1, refresh_y2;
    short *ptr1, *ptr2;
    unsigned char *ptr3;
    ch &= 0x7F; // clear parity bit, which is often set
    if (ch == 0)
        return; // some software will emit character 0 to initialize the teletype.
    if (ch >= '`')
        ch -= 32; // convert to uppercase
    if (ch == 10) {
        scroll = 1;
    } else if (ch == 13) {
        tty_x = 0;
    } else if (ch == 8) { // backspace
        if (tty_x > 0) {
            if (debug)
                --keybufptr;
            --tty_x;
        }
    } else if (ch == 9) {
        for (i = TABWIDTH - (tty_x % TABWIDTH); i > 0; --i)
            tty_out(' ');
    } else if (ch >= ' ') {
        if (text[LINES-1][tty_x] == 0)
            text[LINES-1][tty_x+1] = 0;
        text[LINES-1][tty_x] = ch;
        ++tty_x;
        if (tty_x >= tty_w) {
            tty_x = 0;
            scroll = 1;
        }
    }
    c_cursor1.x = tty_x*5 + 8;
    c_cursor2.x = tty_x*5 + 14;
    if (scroll) {
        for (i = 0; i < LINES-1; ++i)
            memcpy(&text[i][0], &text[i+1][0], LINE_LEN);
        text[LINES-1][0] = 0;
        // messy bit-twiddling for line-by-line scrolling
        refresh_y1 = Win_Height(&form2) - 189 - (tty_scroll * 7);
        refresh_y2 = -189 - (tty_scroll * 7);
        color = tty_scroll + 1;
        ptr1 = &c_hole1;
        ptr2 = &cc_hole1;
        ptr3 = &cd_line1;
        for (i = 0; i < CURSOR_ID-2; i += 5) {
            *(ptr1+4) = refresh_y1;     // left hole y
            *(ptr1+12) = refresh_y1;    // stripe y
            *(ptr1+20) = refresh_y1;    // right hole y
            *(ptr2+2) = refresh_y2;     // left hole y calc rule
            *(ptr2+10) = refresh_y2;    // stripe y calc rule
            *(ptr2+18) = refresh_y2;    // right hole y calc rule
            *(ptr3+2) = (color++ & 2) ? ((COLOR_BLACK << 4) | COLOR_WHITE) : ((COLOR_BLACK << 4) | COLOR_LBLUE);  // line 1 color
            *(ptr3+8) = (color++ & 2) ? ((COLOR_BLACK << 4) | COLOR_WHITE) : ((COLOR_BLACK << 4) | COLOR_LBLUE);  // line 2 color
            refresh_y1 += 14;
            refresh_y2 += 14;
            ptr1 += 40;
            ptr2 += 40;
            ptr3 += 12;
        }
        tty_scroll = (tty_scroll + 1) & 3;
        Win_Redraw(teleID, -1, 0);
    } else {
        Win_Redraw_Area(teleID, -1, 0, (tty_x - 1) * 5 + 7, c_cursor1.y - 4, 21, 15);
    }
    cc_cursor1.xbase = c_cursor1.x;
    cc_cursor2.xbase = c_cursor2.x;
}

void refresh_keys(void) {
    Gfx_Select(canvas2);
    Gfx_Put_Set(keyimg, 9, 12, PUT_SET, power);
    Gfx_Put_Set(keyimg, 9, 25, PUT_SET, lock);
    if (winID)
        Win_Redraw_Area(winID, -1, 0, 9, 101, 9, 22);
    Gfx_Select(canvas1);
}

void refresh_buttons(void) {
    char i, x = 45;
    Gfx_Select(canvas2);
    for (i = 0; i < sizeof(buttons); ++i) {
        Gfx_Put_Set(buttonimg, x, 15, PUT_SET, buttons[i] + button_colors[i]);
        x += 7;
    }
    if (winID)
        Win_Redraw_Area(winID, -1, 0, 45, 103, 182, 17);
    Gfx_Select(canvas1);
}

void draw_light(unsigned char x, unsigned char y, unsigned char val) {
    val = (val & power) + 1;
    Gfx_Pixel(x,     y,     val);
    Gfx_Pixel(x + 1, y,     val);
    Gfx_Pixel(x,     y + 1, val);
    Gfx_Pixel(x + 1, y + 1, val);
}

unsigned char refresh_x1, refresh_y1, refresh_x2, refresh_y2;

void refresh_bounds(unsigned char x1, unsigned char y1, unsigned char x2, unsigned char y2) {
    if (refresh_x1 > x1) refresh_x1 = x1;
    if (refresh_y1 > y1) refresh_y1 = y1;
    if (refresh_x2 < x2) refresh_x2 = x2;
    if (refresh_y2 < y2) refresh_y2 = y2;
}

void light_row(unsigned char x, unsigned char y, unsigned char count, unsigned short val) {
    unsigned char x2;
    x += count*7;
    x2 = x - 6;
    while (count--) {
        x -= 7;
        draw_light(x, y, val & 1);
        val >>= 1;
    }
    refresh_bounds(x, y, x2, y + 1);
}

void refresh_lights(void) {
    signed char i, t;

    refresh_x1 = 255;
    refresh_y1 = 255;
    refresh_x2 = 0;
    refresh_y2 = 0;

    // draw DF/IF
    if (dfield != old_dfield) {
        light_row(48, 34, 3, dfield >> 5);
        old_dfield = dfield;
    }
    if (ifield != old_ifield) {
        light_row(69, 34, 3, ifield >> 5);
        old_ifield = ifield;
    }

    // draw PC
    if (pc != old_pc) {
        light_row(90, 34, 12, pc >> 1);
        old_pc = pc;
    }

    // draw memory address
    if (ma != old_ma) {
        light_row(90, 47, 12, (ma - (unsigned short)core) >> 1);
        old_ma = ma;
    }

    // draw memory buffer
    if (mb != old_mb) {
        light_row(90, 60, 12, mb);
        old_mb = mb;
    }

    // draw accumulator
    if (link != old_link) {
        draw_light(83, 73, link);
        refresh_bounds(83, 73, 84, 74);
        old_link = link;
    }
    if (acc != old_acc) {
        light_row(90, 73, 12, acc);
        old_acc = acc;
    }

    // draw quotient
    if (mq != old_mq) {
        light_row(90, 86, 12, mq);
        old_mq = mq;
    }

    // draw state lights
    if (inst_light != old_inst_light) {
        t = (inst_light & 0x0E);
        for (i = 0; i < 16; i += 2)
            draw_light(193, i*3 + 35, (i == t));
        old_inst_light = inst_light;
        refresh_bounds(193, 35, 194, 78);
    }
    if (state_lights != old_state_lights) {
        for (i = 0; i < 6; ++i)
            draw_light(211, i*6 + 35, (state_lights >> i) & 1);
        old_state_lights = state_lights;
        refresh_bounds(211, 35, 212, 66);
    }
    if (ion != old_ion || run != old_run) {
        draw_light(229, 35, ion & 1);
        draw_light(229, 47, run);
        old_ion = ion;
        old_run = run;
        refresh_bounds(229, 35, 230, 48);
    }

    // refresh
    if (winID != 0 && (refresh_x2 > refresh_x1))
        Win_Redraw_Area(winID, -1, 0, refresh_x1, refresh_y1, refresh_x2 - refresh_x1 + 1, refresh_y2 - refresh_y1 + 1);
}

void rnd16(void) {
    // trivial xorshift PRNG (smaller and faster than libc rand())
    if (rndseed == 0)
        rndseed = 0xACE1;
    rndseed ^= (rndseed << 7);
    rndseed ^= (rndseed >> 9);
    rndseed ^= (rndseed << 8);
}

void random_bits(unsigned char y) {
    unsigned char i;
    for (i = 0; i < 8; ++i) {
        rnd16();
        Gfx_Pixel(16, y, (rndseed & 1) ? COLOR_BLACK : COLOR_WHITE);
        Gfx_Pixel(18, y, (rndseed & 2) ? COLOR_BLACK : COLOR_WHITE);
        Gfx_Pixel(20, y, (rndseed & 4) ? COLOR_BLACK : COLOR_WHITE);
        y += 2;
    }
}

void refresh_tape(void) {
    // select the right canvas
    if (tape_in_next || tape_out_next)
        Gfx_Select(canvas3);

    // tape reader states
    if (tape_in_next == 1) {
        Gfx_Put_Set(tapeimg, 15, 29, PUT_SET, 0);
        Gfx_HLine(15, 53, 8, COLOR_WHITE);
        Gfx_BoxF(15, 58, 22, 64, COLOR_WHITE);
        tape_in_loaded = 0;
    } else if (tape_in_next == 2) {
        random_bits(34);
        if (tape_in_loaded == 0) {
            Gfx_Put_Set(tapeimg, 15, 58, PUT_SET, 1);
            tape_in_loaded = 1;
        } else {
            random_bits(59);
        }
    }
    if (tape_in_next == 3 || tape_in_next == 4) {
        Gfx_HLine(15, 29, 8, COLOR_BLACK);
        Gfx_HLine(15, 30, 8, COLOR_RED);
        Gfx_BoxF(15, 31, 22, 49, COLOR_BLACK);
        Gfx_HLine(15, 53, 8, COLOR_BLACK);
        if (tape_in_next == 4) {
            Gfx_BoxF(15, 58, 22, 76, COLOR_BLACK);
            Gfx_HLine(15, 77, 8, COLOR_RED);
            Gfx_HLine(15, 78, 8, COLOR_BLACK);
        } else {
            Gfx_Put_Set(tapeimg, 15, 58, PUT_SET, 1);
        }
    }

    // tape punch states
    if (tape_out_next == 1) {
        Gfx_Put_Set(tapeimg, 15, 111, PUT_SET, 0);
    } else if (tape_out_next == 2) {
        random_bits(116);
    } else if (tape_out_next == 4) {
        Gfx_HLine(15, 111, 8, COLOR_BLACK);
        Gfx_HLine(15, 112, 8, COLOR_RED);
        Gfx_BoxF(15, 113, 22, 128, COLOR_BLACK);
        Gfx_BoxF(15, 129, 22, 130, COLOR_ORANGE);
        Gfx_HLine(15, 131, 8, COLOR_BLACK);
    }

    // refresh
    if ((tape_in_next || tape_out_next) && winID) {
        Win_Redraw_Area(winID, -1, 0, 247, 29, 8, 103);
        if (tape_in_next == 3)
            tape_in_next = 4;
        else
            tape_in_next = 0;
        tape_out_next = 0;
        Gfx_Select(canvas1);
    }
}

unsigned short debug_oct(void) {
    return ((unsigned short)((keybuf[1] & 0x7F) - '0') << 9) |
           ((unsigned short)((keybuf[2] & 0x7F) - '0') << 6) |
           ((unsigned short)((keybuf[3] & 0x7F) - '0') << 3) |
            (unsigned short)((keybuf[4] & 0x7F) - '0');
}

void debug_cmd(void) {
    unsigned short val = debug_oct();
    unsigned short val2 = val << 1;
    ma = (unsigned short)core + val2;
    switch (*keybuf & 0x5F) {
    case 'A': acc = val; break;
    case 'B': breakpoint = val2 + 1; break;
    case 'D': *(unsigned short*)ma = sr(); break;
    case 'F': mb = pt_rflag | (pt_pflag << 1) | (tty_kflag << 2) | (tty_tflag << 3) | (irq << 4); goto skipmb;
    case 'P': pc = val2 + 1; break;
    case 'S': set_sr(val); break;
    // 'E' occurs automagically as the default.
    }
    mb = *(unsigned short*)ma;
skipmb:
    refresh_lights();
    refresh_buttons();
    keybufptr = keybuf;
    debug = 0;
}

void keypress(unsigned char win, unsigned char ch) {
    if (win == teleID) {
        if (ch == '|') {
            keybufptr = keybuf;
            debug = 1;
        } else if (ch <= 127 && keybufptr < keybuf + BUF_SIZE - 1) {
            if (ch >= '`')
                ch -= 32;   // convert to uppercase
            if (power)
                *keybufptr++ = (ch | 0x80); // DEC convention is to set high bit (parity)
            if (debug) {
                tty_out(ch);
                if (ch == 13) {
                    tty_out(10);
                    debug_cmd();
                }
            }
        }
    }
}

void stop(void) {
    run = 0;
    halt = 0;
    if (!toggle_back)
        active = 0;
    state_lights = 1;
    refresh_lights();
}

void click_button(int mousex, int mousey) {
    unsigned char i = (mousex - 45) / 7;
    unsigned short buttonx = ((mousex - 45) / 7) * 7 + 45;
    if (i > 25)
        return;
    if (mousey <= 112 && buttons[i] == 1)
        buttons[i] = 0;
    else if (mousey > 112 && buttons[i] == 0)
        buttons[i] = 1;
    Gfx_Select(canvas2);
    Gfx_Put_Set(buttonimg, buttonx, 15, PUT_SET, buttons[i] + button_colors[i]);
    Win_Redraw_Area(winID, -1, 0, buttonx, 104, 7, 17);
    Gfx_Select(canvas1);

    if (i > 17 && i < 24) {
        countbyte = 0;
        active = 1;
        toggle_back = i;
    }

    if (power && !lock) {
        switch (i) {
        case 18: // Start
            if (buttons[i] == 1) {
                clearflags();
                run = 1;
                state_lights = 3;
                active = 1;
            }
            break;
        case 19: // Load Add
            if (buttons[i] == 1) {
                pc = sr()*2 + 1;
                dfield = (buttons[0] << 7) | (buttons[1] << 6) | (buttons[2] << 5);
                ifield = (buttons[3] << 7) | (buttons[4] << 6) | (buttons[5] << 5);
                state_lights = 0;
                refresh_lights();
            }
            break;
        case 20: // Dep
            if (buttons[i] == 1) {
                ma = pc + (unsigned short)core;
                mb = sr();
                *(unsigned short*)&core[pc-1] = mb;
                state_lights = 0;
                pc += 2;
                refresh_lights();
            }
            break;
        case 21: // Exam
            if (buttons[i] == 1) {
                ma = pc + (unsigned short)core;
                mb = *(unsigned short*)&core[pc-1];
                state_lights = 0;
                pc += 2;
                refresh_lights();
            }
            break;
        case 22: // Cont
            if (buttons[i] == 1) {
                run = 1;
                active = 1;
                state_lights = 3;
                skip_breakpoint = 1;
            }
            break;
        case 23: // Stop
            if (buttons[i] == 1)
                stop();
            break;
        }
    }
}

void flush_punch(void) {
    File_Write(tape_out, _symbank, tape_out_buf, tape_out_ptr - tape_out_buf);
    tape_out_ptr = tape_out_buf;
}

void attach_input(void) {
    unsigned char i = FileBox(0, "*  ", FILEBOX_OPEN | FILEBOX_FILE, ATTRIB_VOLUME, 128, 8000, &form1);
    if (i == FILEBOX_OK) {
        if (tape_in <= 7) {
            File_Close(tape_in);
            tape_in_next = 4;
            refresh_tape();
        }
        tape_in = File_Open(_symbank, FileBoxPath);
        if (tape_in > 7) {
            MsgBox("Failed to open file.", "", "", COLOR_BLACK, BUTTON_OK | TITLE_WARNING, 0, &form1);
        } else {
            pt_bin = (stricmp(strrchr(FileBoxPath, '.'), ".BIN") == 0);
            pt_rstage = 0;
            tape_in_ptr = tape_in_end + 1; // so it immediately reads
            tape_in_next = 1;
        }
    }
}

void attach_output(void) {
    unsigned char i = FileBox(0, "*  ", FILEBOX_SAVE | FILEBOX_FILE, ATTRIB_VOLUME, 128, 8000, &form1);
    if (i == FILEBOX_OK) {
        if (tape_out <= 7) {
            flush_punch();
            File_Close(tape_out);
            tape_out_next = 4;
            refresh_tape();
        }
        if (Dir_GetAttrib(_symbank, FileBoxPath) != -1) {
            i = MsgBox("Overwrite file?", "", "", COLOR_BLACK, BUTTON_YN | TITLE_CONFIRM, 0, &form1);
            if (i == MSGBOX_NO)
                return;
        }
        tape_out = File_New(_symbank, FileBoxPath, 0);
        tape_out_ptr = tape_out_buf;
        if (tape_out > 7)
            MsgBox("Failed to open file.", "", "", COLOR_BLACK, BUTTON_OK | TITLE_WARNING, 0, &form1);
        else
            tape_out_next = 1;
    }
}

void rim_loader(void) {
    unsigned char i;
    if (tape_in > 7) {
        MsgBox("No input tape attached.", "", "", COLOR_BLACK, BUTTON_OK | TITLE_WARNING, 0, &form1);
        return;
    }
    memset(core, 0, sizeof(core));
    if (pt_bin) {
        memset(core + 7956, 0, 10); // remove image data that might have high bits
        memcpy(core + 7980, binldr, sizeof(binldr));
        *(unsigned short*)(core + 8190) = 05301;
        pc = 8191;
        set_sr(03777); // set SR to 3777 because we're emulating a high-speed tape reader, not ASR 33 that reads tape via TTY (see DEC-08-LBAA-D)
        refresh_buttons();
    } else {
        memcpy(core + 8156, rimldr, sizeof(rimldr));
        pc = 8157;
    }
    power = 1;
    clearflags();
    ifield = 0;
    dfield = 0;
    active = 1;
    run = 1;
    refresh_keys();
}

void quit(void) {
    Counter_Delete(_symbank, &countbyte);
    if (tape_in <= 7)
        File_Close(tape_in);
    if (tape_out <= 7) {
        flush_punch();
        File_Close(tape_out);
    }
    exit();
}

int main(int argc, char *argv[]) {
    unsigned char i;
    unsigned short s;
    int mousex, mousey;

    // display splash screen (since this takes awhile to load!)
    form3.x = (Screen_Width() >> 1) - 32;
    form3.y = (Screen_Height() >> 1) - 6;
    splashID = Win_Open(_symbank, &form3);

    // initialize graphics
    buttons[20] = 1;
    Gfx_Init(holecanvas, 8, 14);
    Gfx_Select(holecanvas);
    Gfx_Prep(holeimg);
    Gfx_Put(holeimg, 0, 0, PUT_SET);
    Gfx_Init(canvas1, 232, 89);
    Gfx_Init(canvas2, 232, 43);
    Gfx_Init(canvas3, 28, 132);
    Gfx_Prep(core);
    Gfx_Prep(lowerimg);
    Gfx_Prep(feedsimg);
    Gfx_Prep_Set(buttonimg);
    Gfx_Prep_Set(keyimg);
    Gfx_Prep_Set(tapeimg);
    Gfx_Select(canvas3);
    Gfx_Put(feedsimg, 0, 0, PUT_SET);
    Gfx_Select(canvas2);
    Gfx_Put(lowerimg, 0, 0, PUT_SET);
    Gfx_Select(canvas1);
    Gfx_Put(core, 0, 0, PUT_SET);
    refresh_keys();
    refresh_buttons();
    refresh_lights();

    // initialize teletype
    memset(text, 0, sizeof(text));

    // set up frame counter
    Counter_Add(_symbank, &countbyte, _sympid, 6); // increments every 6/50th second (about 8 FPS max)

    // open windows
    Win_Close(splashID);
	winID = Win_Open(_symbank, &form1);
	teleID = Win_Open(_symbank, &form2);
	Win_Focus(winID);

    // main loop
	while (1) {
        // handle messages
		_symmsg[0] = 0;
		if (active)
            Msg_Receive(_sympid, -1, _symmsg);
        else
            Msg_Sleep(_sympid, -1, _symmsg);

		if (_symmsg[0]) {
            switch (_symmsg[0]) {
            case MSR_DSK_WCLICK:
                switch (_symmsg[2]) {
                case DSK_ACT_CLOSE: // Alt+F4 or click close
                    quit();
                    break;

                case DSK_ACT_CONTENT: // Regular window click
                    mousex = *(int*)&_symmsg[4];
                    mousey = *(int*)&_symmsg[6];
                    if (_symmsg[3] == DSK_SUB_MLCLICK) {
                        if (mousex >= 47 && mousex <= 227 && mousey >= 103 && mousey <= 119) {
                            click_button(mousex, mousey);
                        } else if (mousex >= 8 && mousex <= 41 && mousey >= 32 && mousey <= 83) {
                            rim_loader();
                        } else if (mousex >= 243 && mousex <= 258 && mousey >= 16 && mousey <= 85) {
                            attach_input();
                        } else if (mousex >= 243 && mousex <= 258 && mousey >= 89 && mousey <= 130) {
                            attach_output();
                        } else if (mousex >= 245 && mousex <= 256 && mousey >= 2 && mousey <= 13) {
                            Dir_PathAdd(0, "PDP8.HLP", FileBoxPath);
                            App_Run(_symbank, FileBoxPath, 0);
                        } else if (mousex >= 10 && mousex <= 18) {
                            if (mousey >= 101 && mousey <= 109) {
                                power ^= 1;
                                refresh_keys();
                                if (power == 1) {
                                    clearflags();
                                    rnd16(); acc = rndseed & 07777;
                                    rnd16(); link = rndseed & 0x01;
                                    rnd16(); ir = rndseed & 07777;
                                    rnd16(); ifield = (rndseed & 01) << 5;
                                    rnd16(); dfield = (rndseed & 01) << 5;
                                    rnd16(); ma = (rndseed & 07777) * 2 + (unsigned short)core;
                                    rnd16(); mb = rndseed & 07777;
                                    rnd16(); mq = rndseed & 07777;
                                    rnd16(); inst_light = rndseed & 017;
                                    rnd16(); state_lights = 1 << (rndseed & 7);
                                    memset(core, 0, sizeof(core));
                                    for (pc = 0; pc < 4096; pc += 2) { // it takes awhile to load the full core with random values, so we give the illusion by loading just the start.
                                        rnd16();
                                        *(unsigned short*)&core[pc] = rndseed & 07777;
                                    }
                                    rnd16(); pc = rndseed & 07777;
                                } else {
                                    run = 0;
                                    active = 0;
                                    old_acc = 0;    // to force refresh
                                    old_dfield = 0;
                                    old_ifield = 0;
                                    old_inst_light = 0;
                                    old_ion = 0;
                                    old_link = 0;
                                    old_ma = 0;
                                    old_mb = 0;
                                    old_mq = 0;
                                    old_pc = 0;
                                    old_run = 0;
                                    old_state_lights = 0;
                                }
                                refresh_lights();
                            } else if (mousey >= 114 && mousey <= 122) {
                                lock ^= 1;
                                refresh_keys();
                                refresh_lights();
                            }
                        }
                    } else if (_symmsg[3] == DSK_SUB_KEY) {
                        keypress(_symmsg[1], _symmsg[4]);
                    }
                    break;

                case DSK_ACT_KEY:
                    keypress(_symmsg[1], _symmsg[4]);
                    break;
                }
                break;

            case MSR_DSK_WRESIZ:
                if (_symmsg[1] == teleID)
                    tty_w = (Win_Width(&form2) - 19) / 5;
                break;
            }
		}

		// run code
        if (run == 1) {
            if (skip_breakpoint) {
                // step into current instruction, skipping breakpoint
                old_breakpoint = breakpoint;
                breakpoint = 0xFFFF;
                do_inst();
                breakpoint = old_breakpoint;
                skip_breakpoint = 0;
            } else if (ion == 3) {
                // single instruction, finishing ION turn-on
                do_inst();
                ion &= 1; // can't just do ion = 1 because the intervening instruction may have been IOF
            } else if (buttons[25]) {
                // single instruction
                do_inst();
            } else {
                // multiple instructions
                do_multiple();
            }
            if (halt || buttons[25])
                stop();
        }

        if (run) {
            // handle I/O
            if (pt_rgo == 1 && tape_in <= 7) {
                s = 1;
                if (tape_in_ptr >= tape_in_end) {
                    s = File_Read(tape_in, _symbank, tape_in_buf, BUF_SIZE);
                    memset(tape_in_buf + s, 0x80, BUF_SIZE - s); // fill remainder with 0x80 (trailer)
                    tape_in_ptr = tape_in_buf;
                }
                if (s == 0 || (pt_rstage >= 3 && *tape_in_ptr == 0x1A)) {
                    // out of data, or hit 0x1F (AMSDOS EOF) after trailer
                    // FIXME: only trap 0x1A on final block?
                    File_Close(tape_in);
                    tape_in = 8;
                    tape_in_next = 3;
                } else {
                    pt_rbuf = *tape_in_ptr++;
                    pt_rflag = 1;
                    pt_rgo = 0;
                    irq |= IRQ_PTR;
                    if (tape_in_next == 0)
                        tape_in_next = 2;
                    if ((pt_rstage & 1) != (pt_rbuf >> 7)) // tracks header/trailer stage, based on bit 7 (channel 8); pt_rstage == 2 is the main data stream
                        ++pt_rstage;
                }
            }
            if (pt_pgo == 1 && tape_out <= 7) {
                if (tape_out_ptr >= tape_out_end)
                    flush_punch();
                *tape_out_ptr++ = pt_pbuf;
                pt_pflag = 1;
                pt_pgo = 0;
                irq |= IRQ_PTP;
                tape_out_next = 2;
            }
            if (tty_tgo == 1) {
                tty_out(tty_tbuf);
                tty_tflag = 1;
                tty_tbuf = 0;
                tty_tgo = 0;
                irq |= IRQ_TTYP;
            }
            if (tty_kflag == 0 && keybufptr > keybuf && debug == 0) {
                tty_kbuf = *(--keybufptr);
                tty_kflag = 1;
                irq |= IRQ_TTYK;
            }

            // handle interrupts
            if (ion == 1 && ion_mmu == 0 && irq != 0) {
                *(unsigned short*)core = pc >> 1;
                ion = 0;
                intbuf = (dfield >> 5) | (ifield >> 2);
                ifield = 0;
                dfield = 0;
                pc = 3;
            }
        }

		// handle UI counters
		if (countbyte > 0 && (run == 1 || tape_in_next != 0 || tape_out_next != 0)) {
            refresh_lights();
            refresh_tape();
            if (toggle_back)
                countbyte = 2;
            else
                countbyte = 0;
		}
        if (toggle_back > 0 && countbyte > 1) {
            click_button(toggle_back*7 + 47, toggle_back == 20 ? 115 : 107);
            toggle_back = 0;
            if (!run)
                active = 0;
        }
	}
}
